<?php
namespace views;

use Views\LoginModalHtml;

class PublicSearchHtml
{
    /**
     * Afficher la page de résultats de recherche
     */
    public static function displaySearchResults($searchObj)
    {
        $results = $searchObj->getResults();
        $totalResults = $searchObj->getTotalResults();
        $currentPage = $searchObj->getCurrentPage();
        $totalPages = $searchObj->getTotalPages();
        $searchTitle = $searchObj->getSearchTitle();
        $activeFilters = $searchObj->getActiveFilters();
        $searchParams = $searchObj->getSearchParams();
        
        ?>
        <!-- Container principal avec résultats et sidebar -->
        <div class="search-results-wrapper">
            <!-- Résultats de recherche -->
            <div class="search-results-container">
            <div class="container container-bordered">
                <!-- En-tête de résultats de recherche -->
                <div class="search-header-content mb-3">
                    <div class="search-title-section">
                        <h1 class="search-title"><?= htmlspecialchars($searchTitle) ?></h1>
                        <p class="search-count">
                            <i class="fas fa-home"></i>
                            <strong><?= number_format($totalResults, 0, ',', ' ') ?></strong> 
                            <?= $totalResults > 1 ? 'biens trouvés' : 'bien trouvé' ?>
                        </p>
                    </div>
                </div>
                <?php if (empty($results)): ?>
                    <!-- Aucun résultat -->
                    <div class="no-results">
                        <div class="no-results-icon">
                            <i class="fas fa-search"></i>
                        </div>
                        <h2>Aucun bien trouvé</h2>
                        <p>Essayez de modifier vos critères de recherche pour obtenir plus de résultats.</p>
                        <a href="index.php" class="btn btn-primary">
                            <i class="fas fa-arrow-left"></i> Retour à l'accueil
                        </a>
                    </div>
                <?php else: ?>
                    <!-- Grille de résultats -->
                    <div class="results-grid">
                        <?php foreach ($results as $annonce): ?>
                            <?= self::generatePropertyCard($annonce) ?>
                        <?php endforeach; ?>
                    </div>

                    <!-- Pagination -->
                    <?php if ($totalPages > 1): ?>
                    <div class="pagination-container">
                        <?= self::generatePagination($currentPage, $totalPages, $searchObj->getSearchParams()) ?>
                    </div>
                    <?php endif; ?>
                <?php endif; ?>
            </div>
            </div>
            
            <!-- Sidebar Promoteurs vedettes -->
            <div class="promoters-sidebar">
                <div class="promoters-card">
                    <h3 class="promoters-title">
                        <i class="fas fa-star text-warning me-2"></i>
                        Promoteurs vedettes
                    </h3>
                    <?php 
                    // TODO: Récupérer les promoteurs vedettes depuis la base de données
                    // SELECT * FROM users WHERE is_featured = 1 AND user_type = 'promoter' LIMIT 5
                    ?>
                    <div class="promoters-list">
                        <!-- Les promoteurs seront ajoutés ici via PHP -->
                        <p class="text-muted text-center py-3">
                            <i class="fas fa-info-circle me-1"></i>
                            Les promoteurs vedettes seront affichés ici
                        </p>
                    </div>
                </div>
            </div>
        </div>

        <?php
        // Afficher le modal de connexion et son script
        LoginModalHtml::renderLoginModal();
        LoginModalHtml::renderLoginModalScript('.btn-favorite');
        ?>
        <?php
    }

    /**
     * Générer une carte de propriété
     */
    private static function generatePropertyCard($annonce)
    {
        // Image par défaut si pas de photo
        $imageUrl = !empty($annonce['photo_principale']) 
            ? htmlspecialchars($annonce['photo_principale']) 
            : 'https://images.unsplash.com/photo-1560518883-ce09059eeffa?w=800&h=600&fit=crop';

        // Badge de transaction
        $transactionBadge = '';
        if (!empty($annonce['type_transaction'])) {
            $badgeClass = strtolower($annonce['type_transaction']) === 'vente' ? 'badge-sale' : 'badge-rent';
            $transactionBadge = '<span class="property-badge ' . $badgeClass . '">' . 
                                htmlspecialchars($annonce['type_transaction']) . '</span>';
        }

        // Prix formaté
        $price = self::formatPrice($annonce['prix']);
        $priceUnit = strtolower($annonce['type_transaction']) === 'location' ? '/mois' : '';

        // Caractéristiques
        $features = [];
        if (!empty($annonce['surface'])) {
            $features[] = '<span><i class="fas fa-ruler-combined"></i> ' . 
                         number_format($annonce['surface'], 0, ',', ' ') . ' m²</span>';
        }
        if (!empty($annonce['nb_pieces'])) {
            $features[] = '<span><i class="fas fa-th-large"></i> ' . 
                         $annonce['nb_pieces'] . ' pièces</span>';
        }
        if (!empty($annonce['nb_chambres'])) {
            $features[] = '<span><i class="fas fa-bed"></i> ' . 
                         $annonce['nb_chambres'] . ' chambres</span>';
        }
        if (!empty($annonce['nb_salles_bain'])) {
            $features[] = '<span><i class="fas fa-bath"></i> ' . 
                         $annonce['nb_salles_bain'] . ' salle' . ($annonce['nb_salles_bain'] > 1 ? 's' : '') . ' de bain</span>';
        }

        // Localisation
        $location = [];
        if (!empty($annonce['ville'])) $location[] = $annonce['ville'];
        if (!empty($annonce['departement'])) $location[] = $annonce['departement'];
        $locationStr = implode(', ', $location);

        // Badge mise en avant
        $featuredBadge = $annonce['mise_en_avant'] == 1 
            ? '<span class="featured-badge"><i class="fas fa-star"></i> À la une</span>' 
            : '';

        // Nombre de photos
        $photosCount = $annonce['nb_photos'] > 0 
            ? '<span class="photos-count"><i class="fas fa-camera"></i> ' . $annonce['nb_photos'] . '</span>'
            : '';

        ob_start();
        ?>
        <div class="property-card animate-on-scroll" data-annonce-id="<?= $annonce['id'] ?>" onclick="window.location.href='index.php?page=annonce&id=<?= $annonce['id'] ?>'" style="cursor: pointer;">
            <div class="property-image">
                <img src="<?= $imageUrl ?>" alt="<?= htmlspecialchars($annonce['titre']) ?>" loading="lazy">
                <div class="property-badges">
                    <?= $featuredBadge ?>
                    <?= $transactionBadge ?>
                </div>
                <?= $photosCount ?>
                <button class="btn-favorite" data-annonce-id="<?= $annonce['id'] ?>" title="Ajouter aux favoris" onclick="event.stopPropagation();">
                    <i class="far fa-heart"></i>
                </button>
            </div>
            
            <div class="property-content">
                <div class="property-price-top">
                    <?= $price ?> <?= $priceUnit ?>
                </div>
                
                <div class="property-header">
                    <h3 class="property-title">
                        <a href="index.php?page=annonce&id=<?= $annonce['id'] ?>" onclick="event.stopPropagation();">
                            <?= htmlspecialchars($annonce['titre']) ?>
                        </a>
                    </h3>
                </div>

                <?php if (!empty($locationStr)): ?>
                <div class="property-location">
                    <i class="fas fa-map-marker-alt"></i>
                    <?= htmlspecialchars($locationStr) ?>
                </div>
                <?php endif; ?>

                <?php if (!empty($features)): ?>
                <div class="property-features">
                    <?= implode('', $features) ?>
                </div>
                <?php endif; ?>

                <?php if (!empty($annonce['description'])): ?>
                <p class="property-description">
                    <?= htmlspecialchars(mb_substr($annonce['description'], 0, 120)) ?>...
                </p>
                <?php endif; ?>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * Générer la pagination
     */
    private static function generatePagination($currentPage, $totalPages, $searchParams)
    {
        $pagination = '<nav class="pagination" aria-label="Pagination des résultats">';
        $pagination .= '<ul class="pagination-list">';

        // Bouton précédent
        if ($currentPage > 1) {
            $prevPage = $currentPage - 1;
            $prevUrl = self::buildSearchUrl($searchParams, $prevPage);
            $pagination .= '<li><a href="' . $prevUrl . '" class="pagination-link pagination-prev" aria-label="Page précédente">';
            $pagination .= '<i class="fas fa-chevron-left"></i> Précédent</a></li>';
        }

        // Numéros de page
        $range = 2; // Nombre de pages à afficher de chaque côté
        $start = max(1, $currentPage - $range);
        $end = min($totalPages, $currentPage + $range);

        // Première page
        if ($start > 1) {
            $url = self::buildSearchUrl($searchParams, 1);
            $pagination .= '<li><a href="' . $url . '" class="pagination-link">1</a></li>';
            if ($start > 2) {
                $pagination .= '<li><span class="pagination-ellipsis">...</span></li>';
            }
        }

        // Pages du milieu
        for ($i = $start; $i <= $end; $i++) {
            $url = self::buildSearchUrl($searchParams, $i);
            $activeClass = $i === $currentPage ? ' active' : '';
            $pagination .= '<li><a href="' . $url . '" class="pagination-link' . $activeClass . '">' . $i . '</a></li>';
        }

        // Dernière page
        if ($end < $totalPages) {
            if ($end < $totalPages - 1) {
                $pagination .= '<li><span class="pagination-ellipsis">...</span></li>';
            }
            $url = self::buildSearchUrl($searchParams, $totalPages);
            $pagination .= '<li><a href="' . $url . '" class="pagination-link">' . $totalPages . '</a></li>';
        }

        // Bouton suivant
        if ($currentPage < $totalPages) {
            $nextPage = $currentPage + 1;
            $nextUrl = self::buildSearchUrl($searchParams, $nextPage);
            $pagination .= '<li><a href="' . $nextUrl . '" class="pagination-link pagination-next" aria-label="Page suivante">';
            $pagination .= 'Suivant <i class="fas fa-chevron-right"></i></a></li>';
        }

        $pagination .= '</ul>';
        $pagination .= '</nav>';

        return $pagination;
    }

    /**
     * Construire l'URL de recherche avec pagination
     */
    private static function buildSearchUrl($params, $page = null)
    {
        $urlParams = [];
        
        if (!empty($params['city_ids']) && is_array($params['city_ids'])) {
            foreach ($params['city_ids'] as $cityId) {
                $urlParams[] = 'city_ids[]=' . urlencode($cityId);
            }
        }
        
        if (!empty($params['cities'])) {
            $urlParams[] = 'cities=' . urlencode($params['cities']);
        }
        
        if (!empty($params['transaction'])) {
            $urlParams[] = 'transaction=' . urlencode($params['transaction']);
        }
        
        if (!empty($params['property'])) {
            $urlParams[] = 'property=' . urlencode($params['property']);
        }
        
        if (!empty($params['price_min'])) {
            $urlParams[] = 'price_min=' . urlencode($params['price_min']);
        }
        
        if (!empty($params['price_max'])) {
            $urlParams[] = 'price_max=' . urlencode($params['price_max']);
        }
        
        if ($page !== null && $page > 1) {
            $urlParams[] = 'p=' . intval($page);
        }
        
        return 'index.php?page=search&' . implode('&', $urlParams);
    }

    /**
     * Formater le prix
     */
    private static function formatPrice($price)
    {
        if ($price >= 1000000) {
            return number_format($price / 1000000, 2, ',', ' ') . ' M DA';
        } elseif ($price >= 1000) {
            return number_format($price / 1000, 0, ',', ' ') . ' K DA';
        }
        return number_format($price, 0, ',', ' ') . ' DA';
    }
}
