<?php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
require_once "config.php";

header('Content-Type: application/json');

// Vérifier si la requête est POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Méthode non autorisée']);
    exit;
}

// Récupérer et valider les données
$prenom = trim($_POST['prenom'] ?? '');
$nom = trim($_POST['nom'] ?? '');
$email = trim($_POST['email'] ?? '');
$telephone = trim($_POST['telephone'] ?? '');
$password = $_POST['password'] ?? '';
$confirmPassword = $_POST['confirm_password'] ?? '';
$terms = isset($_POST['accept_terms']);

// Validation côté serveur
$errors = [];

if (empty($prenom) || strlen($prenom) < 2) {
    $errors[] = 'Le prénom doit contenir au moins 2 caractères';
}

if (empty($nom) || strlen($nom) < 2) {
    $errors[] = 'Le nom doit contenir au moins 2 caractères';
}

if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
    $errors[] = 'Adresse email invalide';
}

if (empty($telephone) || !preg_match('/^[0-9\s\-\+\(\)]{8,}$/', $telephone)) {
    $errors[] = 'Numéro de téléphone invalide';
}

if (empty($password) || strlen($password) < 6) {
    $errors[] = 'Le mot de passe doit contenir au moins 6 caractères';
}

if ($password !== $confirmPassword) {
    $errors[] = 'Les mots de passe ne correspondent pas';
}

if (!$terms) {
    $errors[] = 'Vous devez accepter les conditions d\'utilisation';
}

// Vérifier s'il y a des erreurs
if (!empty($errors)) {
    echo json_encode(['success' => false, 'message' => implode(', ', $errors)]);
    exit;
}

try {
    // Vérifier si l'email existe déjà
    $stmt = $pdo->prepare("SELECT id_user FROM users WHERE email = ?");
    $stmt->execute([$email]);
    
    if ($stmt->fetch()) {
        echo json_encode(['success' => false, 'message' => 'Cette adresse email est déjà utilisée']);
        exit;
    }
    
    // Hacher le mot de passe
    $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
    
    // Insérer le nouvel utilisateur
    $nomComplet = $prenom . ' ' . $nom;
    $stmt = $pdo->prepare("
        INSERT INTO users (nom, email, telephone, mot_de_passe, role_id, actif) 
        VALUES (?, ?, ?, ?, ?, 1)
    ");
    
    // Role ID 3 = Client (utilisateur standard)
    $roleId = 3;
    $stmt->execute([$nomComplet, $email, $telephone, $hashedPassword, $roleId]);
    
    $userId = $pdo->lastInsertId();
    
    // Créer une session pour l'utilisateur
    $_SESSION['user_id'] = $userId;
    $_SESSION['user_name'] = $nomComplet;
    $_SESSION['user_email'] = $email;
    $_SESSION['role_id'] = $roleId;
    $_SESSION['login_time'] = time();
    
    echo json_encode([
        'success' => true, 
        'message' => 'Compte créé avec succès ! Redirection en cours...',
        'redirect' => '/taghilt/visiteur/'
    ]);
    
} catch (PDOException $e) {
    error_log("Erreur d'inscription: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => 'Erreur lors de la création du compte. Veuillez réessayer.']);
}
?>