/**
 * JavaScript pour la page de résultats de recherche
 * Gestion des filtres, favoris, et interactivité
 */

(function() {
    'use strict';

    // Au chargement de la page
    document.addEventListener('DOMContentLoaded', function() {
        initFilterRemoval();
        initFavorites();
        initPropertyCards();
    });

    /**
     * Initialiser la suppression des filtres
     */
    function initFilterRemoval() {
        // Supprimer un filtre individuel
        const filterRemoveButtons = document.querySelectorAll('.filter-remove');
        filterRemoveButtons.forEach(button => {
            button.addEventListener('click', function(e) {
                e.preventDefault();
                const filterTag = this.closest('.filter-tag');
                const filterType = filterTag.dataset.filterType;
                removeFilter(filterType);
            });
        });

        // Effacer tous les filtres
        const clearAllButton = document.querySelector('.clear-all-filters');
        if (clearAllButton) {
            clearAllButton.addEventListener('click', function(e) {
                e.preventDefault();
                clearAllFilters();
            });
        }
    }

    /**
     * Supprimer un filtre et recharger la page
     */
    function removeFilter(filterType) {
        const url = new URL(window.location.href);
        const params = new URLSearchParams(url.search);

        switch(filterType) {
            case 'city':
                // Supprimer toutes les villes pour l'instant
                // TODO: Permettre la suppression d'une ville spécifique
                params.delete('city_ids[]');
                params.delete('cities');
                break;
            case 'transaction':
                params.delete('transaction');
                break;
            case 'property':
                params.delete('property');
                break;
            case 'price':
                params.delete('price_min');
                params.delete('price_max');
                break;
        }

        // Retour à la première page
        params.delete('page');

        // Rediriger vers la nouvelle URL
        window.location.href = 'index.php?page=search&' + params.toString();
    }

    /**
     * Effacer tous les filtres
     */
    function clearAllFilters() {
        // Rediriger vers la page d'accueil ou une recherche vide
        if (confirm('Voulez-vous effacer tous les filtres et retourner à l\'accueil ?')) {
            window.location.href = 'index.php';
        }
    }

    /**
     * Initialiser la gestion des favoris
     */
    function initFavorites() {
        const favoriteButtons = document.querySelectorAll('.btn-favorite');
        favoriteButtons.forEach(button => {
            button.addEventListener('click', function(e) {
                e.preventDefault();
                e.stopPropagation();
                toggleFavorite(this);
            });
        });
    }

    /**
     * Basculer l'état favori
     */
    function toggleFavorite(button) {
        const propertyCard = button.closest('.property-card');
        const annonceId = propertyCard.dataset.annonceId;
        const icon = button.querySelector('i');

        // Basculer l'icône
        if (icon.classList.contains('far')) {
            // Ajouter aux favoris
            icon.classList.remove('far');
            icon.classList.add('fas');
            button.classList.add('active');
            
            // Animation
            button.style.animation = 'heartBeat 0.5s ease';
            setTimeout(() => {
                button.style.animation = '';
            }, 500);

            // TODO: Envoyer la requête AJAX pour ajouter aux favoris
            console.log('Ajout aux favoris:', annonceId);
            showNotification('Ajouté aux favoris', 'success');
        } else {
            // Retirer des favoris
            icon.classList.remove('fas');
            icon.classList.add('far');
            button.classList.remove('active');

            // TODO: Envoyer la requête AJAX pour retirer des favoris
            console.log('Retrait des favoris:', annonceId);
            showNotification('Retiré des favoris', 'info');
        }
    }

    /**
     * Initialiser les interactions sur les cartes de propriétés
     */
    function initPropertyCards() {
        const propertyCards = document.querySelectorAll('.property-card');
        
        propertyCards.forEach(card => {
            // Animation au survol
            card.addEventListener('mouseenter', function() {
                this.style.transform = 'translateY(-8px)';
            });

            card.addEventListener('mouseleave', function() {
                this.style.transform = 'translateY(0)';
            });
        });
    }

    /**
     * Afficher une notification
     */
    function showNotification(message, type = 'info') {
        // Créer l'élément de notification
        const notification = document.createElement('div');
        notification.className = `search-notification notification-${type}`;
        notification.innerHTML = `
            <i class="fas fa-${type === 'success' ? 'check-circle' : 'info-circle'}"></i>
            <span>${message}</span>
        `;

        // Ajouter au DOM
        document.body.appendChild(notification);

        // Animer l'apparition
        setTimeout(() => {
            notification.classList.add('show');
        }, 10);

        // Retirer après 3 secondes
        setTimeout(() => {
            notification.classList.remove('show');
            setTimeout(() => {
                notification.remove();
            }, 300);
        }, 3000);
    }

    // Styles pour les notifications (injectés dynamiquement)
    const style = document.createElement('style');
    style.textContent = `
        .search-notification {
            position: fixed;
            top: 20px;
            right: 20px;
            background: white;
            padding: 15px 20px;
            border-radius: 10px;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.15);
            display: flex;
            align-items: center;
            gap: 10px;
            opacity: 0;
            transform: translateX(400px);
            transition: all 0.3s ease;
            z-index: 10000;
            font-weight: 500;
        }

        .search-notification.show {
            opacity: 1;
            transform: translateX(0);
        }

        .notification-success {
            border-left: 4px solid #28a745;
        }

        .notification-success i {
            color: #28a745;
        }

        .notification-info {
            border-left: 4px solid #17a2b8;
        }

        .notification-info i {
            color: #17a2b8;
        }

        @keyframes heartBeat {
            0%, 100% { transform: scale(1); }
            25% { transform: scale(1.3); }
            50% { transform: scale(1.1); }
            75% { transform: scale(1.2); }
        }

        .btn-favorite.active {
            background: #fff5f5;
        }
    `;
    document.head.appendChild(style);

})();
