/**
 * JavaScript pour le formulaire de recherche moderne
 * Gestion de l'autocomplétion, du slider de prix et des interactions
 */

(function() {
    'use strict';

    // Configuration
    const CONFIG = {
        apiUrl: 'api_localites.php',
        minSearchLength: 2,
        debounceDelay: 300,
        maxPrice: 50000000, // 50 millions DA
        priceStep: 1000000  // 1 million DA
    };

    // État de l'application
    const state = {
        selectedCities: [], // Array pour stocker les villes sélectionnées
        priceMin: 0,
        priceMax: CONFIG.maxPrice,
        searchTimeout: null
    };

    // Éléments DOM
    const elements = {
        cityInput: document.getElementById('city-search'),
        autocompleteContainer: document.getElementById('city-autocomplete'),
        selectedCitiesContainer: document.getElementById('selected-cities-container'),
        selectedCities: document.getElementById('selected-cities'),
        priceToggle: document.getElementById('price-toggle'),
        pricePanel: document.getElementById('price-panel'),
        priceDisplay: document.getElementById('price-display'),
        priceMinInput: document.getElementById('price-min'),
        priceMaxInput: document.getElementById('price-max'),
        priceSlider: document.getElementById('price-slider'),
        searchForm: document.getElementById('search-form'),
        transactionType: document.getElementById('transaction_type'),
        propertyType: document.getElementById('property_type')
    };

    /**
     * Initialisation
     */
    function init() {
        if (!elements.cityInput) return;

        setupCityAutocomplete();
        setupPricePanel();
        setupPriceSlider();
        setupFormSubmit();
        setupPresetButtons();
        setupSessionCitiesLoader();
    }

    /**
     * Configuration du chargement des villes depuis la session
     */
    function setupSessionCitiesLoader() {
        // Écouter l'événement personnalisé pour charger les villes de la session
        document.addEventListener('addCityFromSession', function(e) {
            const city = e.detail;
            console.log('📥 Chargement ville de la session:', city.label);
            
            // Vérifier si la ville n'est pas déjà dans l'état
            const alreadyExists = state.selectedCities.some(c => c.id === city.id);
            if (!alreadyExists) {
                // Ajouter la ville à l'état
                state.selectedCities.push(city);
                
                // Créer le tag visuel
                addCityTag(city);
                
                // Afficher le conteneur si c'est la première ville
                if (state.selectedCities.length === 1) {
                    elements.selectedCitiesContainer.style.display = 'block';
                }
            }
        });
    }

    /**
     * Configuration de l'autocomplétion des villes
     */
    function setupCityAutocomplete() {
        // Événement input sur le champ ville
        elements.cityInput.addEventListener('input', function(e) {
            const query = e.target.value.trim();

            // Clear timeout précédent
            clearTimeout(state.searchTimeout);

            if (query.length < CONFIG.minSearchLength) {
                hideAutocomplete();
                elements.autocompleteContainer.innerHTML = '';
                return;
            }

            // Fermer le panel Budget si ouvert
            if (elements.pricePanel.style.display === 'block') {
                elements.pricePanel.style.display = 'none';
                elements.priceToggle.classList.remove('active');
            }

            // Debounce la recherche
            state.searchTimeout = setTimeout(() => {
                searchCities(query);
            }, CONFIG.debounceDelay);
        });

        // Fermer l'autocomplete en cliquant ailleurs
        document.addEventListener('click', function(e) {
            if (!elements.cityInput.contains(e.target) && !elements.autocompleteContainer.contains(e.target)) {
                hideAutocomplete();
            }
        });

        // Focus sur le champ - fermer Budget si ouvert et vider l'autocomplete
        elements.cityInput.addEventListener('focus', function() {
            // Fermer le panel Budget
            if (elements.pricePanel.style.display === 'block') {
                elements.pricePanel.style.display = 'none';
                elements.priceToggle.classList.remove('active');
            }
            
            // TOUJOURS cacher et vider l'autocomplete au focus
            hideAutocomplete();
            elements.autocompleteContainer.innerHTML = '';
        });
        
        // Click sur le champ - vider et cacher l'autocomplete
        elements.cityInput.addEventListener('click', function() {
            // TOUJOURS cacher et vider l'autocomplete au click
            hideAutocomplete();
            elements.autocompleteContainer.innerHTML = '';
        });
    }

    /**
     * Recherche des villes via l'API
     */
    function searchCities(query) {
        console.log('🔍 Recherche pour:', query);
        
        // Afficher loading
        showAutocomplete();
        elements.autocompleteContainer.innerHTML = `
            <div class="autocomplete-loading">
                <i class="fas fa-spinner"></i> Recherche...
            </div>
        `;

        // Appel API
        const apiUrl = `${CONFIG.apiUrl}?action=search&q=${encodeURIComponent(query)}`;
        console.log('📡 URL API:', apiUrl);
        
        fetch(apiUrl)
            .then(response => {
                console.log('📥 Réponse reçue:', response.status);
                return response.json();
            })
            .then(data => {
                console.log('✅ Données reçues:', data);
                displayAutocompleteResults(data, query);
            })
            .catch(error => {
                console.error('❌ Erreur recherche ville:', error);
                elements.autocompleteContainer.innerHTML = `
                    <div class="autocomplete-loading text-danger">
                        <i class="fas fa-exclamation-circle"></i> Erreur de recherche
                    </div>
                `;
            });
    }

    /**
     * Afficher les résultats de l'autocomplétion
     */
    function displayAutocompleteResults(results, query) {
        console.log('🎨 Affichage de', results.length, 'résultats');
        
        if (results.length === 0) {
            elements.autocompleteContainer.innerHTML = `
                <div class="autocomplete-loading">
                    <i class="fas fa-info-circle"></i> Aucune ville trouvée
                </div>
            `;
            return;
        }

        elements.autocompleteContainer.innerHTML = '';

        results.forEach(result => {
            const item = document.createElement('div');
            item.className = 'autocomplete-item';
            
            // Icône selon le type
            const icon = result.type === 'wilaya' ? 'fa-map-marked-alt' : 
                        result.type === 'commune' ? 'fa-map-marker-alt' : 'fa-map-pin';

            item.innerHTML = `
                <i class="fas ${icon}"></i>
                <div class="autocomplete-item-text">
                    <strong>${result.label}</strong>
                    <small>${result.type === 'wilaya' ? 'Wilaya' : 'Commune'} - ${result.code_postal || result.code_wilaya}</small>
                </div>
            `;

            item.addEventListener('click', () => selectCity(result));
            elements.autocompleteContainer.appendChild(item);
        });

        showAutocomplete();
    }

    /**
     * Sélectionner une ville
     */
    function selectCity(city) {
        // Vérifier si la ville n'est pas déjà sélectionnée
        const alreadySelected = state.selectedCities.some(c => c.id === city.id);
        if (alreadySelected) {
            console.log('⚠️ Ville déjà sélectionnée:', city.label);
            elements.cityInput.value = '';
            hideAutocomplete();
            return;
        }

        // Ajouter la ville à la liste
        state.selectedCities.push(city);
        console.log('✅ Ville ajoutée:', city.label, '| Total:', state.selectedCities.length);

        // Créer le tag visuel
        addCityTag(city);

        // Afficher le conteneur si c'est la première ville
        if (state.selectedCities.length === 1) {
            elements.selectedCitiesContainer.style.display = 'block';
        }

        // Effacer l'input et masquer l'autocomplete
        elements.cityInput.value = '';
        hideAutocomplete();
    }

    /**
     * Ajouter un tag de ville
     */
    function addCityTag(city) {
        const tag = document.createElement('div');
        tag.className = 'city-tag';
        tag.dataset.cityId = city.id;
        
        tag.innerHTML = `
            <span class="city-tag-name">
                <i class="fas fa-map-marker-alt"></i>
                ${city.label}
            </span>
            <button type="button" class="city-tag-remove" data-city-id="${city.id}" title="Supprimer">
                ×
            </button>
        `;

        // Event listener pour le bouton de suppression
        const removeBtn = tag.querySelector('.city-tag-remove');
        removeBtn.addEventListener('click', function() {
            removeCity(city.id);
        });

        elements.selectedCities.appendChild(tag);
    }

    /**
     * Supprimer une ville sélectionnée
     */
    function removeCity(cityId) {
        console.log('🗑️ Suppression ville ID:', cityId);

        // Retirer de l'état
        state.selectedCities = state.selectedCities.filter(c => c.id !== cityId);

        // Retirer le tag du DOM
        const tag = elements.selectedCities.querySelector(`[data-city-id="${cityId}"]`);
        if (tag) {
            tag.style.animation = 'tagDisappear 0.3s ease-out';
            setTimeout(() => {
                tag.remove();
                
                // Masquer le conteneur si plus aucune ville
                if (state.selectedCities.length === 0) {
                    elements.selectedCitiesContainer.style.display = 'none';
                }
            }, 300);
        }
    }

    /**
     * Afficher/cacher l'autocomplete (style comme price-panel)
     */
    function showAutocomplete() {
        console.log('👁️ Affichage autocomplete');
        elements.autocompleteContainer.style.display = 'block';
    }

    function hideAutocomplete() {
        console.log('🙈 Masquage autocomplete');
        elements.autocompleteContainer.style.display = 'none';
    }

    /**
     * Configuration du panel de prix
     */
    function setupPricePanel() {
        elements.priceToggle.addEventListener('click', function(e) {
            e.preventDefault();
            const isActive = elements.pricePanel.style.display === 'block';
            
            if (isActive) {
                elements.pricePanel.style.display = 'none';
                elements.priceToggle.classList.remove('active');
            } else {
                // Fermer l'autocomplete si ouvert
                hideAutocomplete();
                
                elements.pricePanel.style.display = 'block';
                elements.priceToggle.classList.add('active');
            }
        });

        // Synchroniser les inputs avec l'affichage
        elements.priceMinInput.addEventListener('input', updatePriceDisplay);
        elements.priceMaxInput.addEventListener('input', updatePriceDisplay);
    }

    /**
     * Configuration du slider de prix (double slider)
     */
    function setupPriceSlider() {
        const sliderMin = document.getElementById('price-slider-min');
        const sliderMax = document.getElementById('price-slider-max');
        const sliderRange = document.getElementById('price-slider-range');
        const labelMin = document.getElementById('slider-label-min');
        const labelMax = document.getElementById('slider-label-max');

        if (!sliderMin || !sliderMax || !sliderRange) return;

        // Fonction pour mettre à jour la barre de couleur
        function updateSliderRange() {
            const min = parseInt(sliderMin.value);
            const max = parseInt(sliderMax.value);
            
            const percentMin = (min / CONFIG.maxPrice) * 100;
            const percentMax = (max / CONFIG.maxPrice) * 100;
            
            sliderRange.style.left = percentMin + '%';
            sliderRange.style.right = (100 - percentMax) + '%';
            
            // Mettre à jour les labels
            labelMin.textContent = formatPrice(min);
            labelMax.textContent = formatPrice(max);
        }

        // Gérer le z-index dynamiquement pour les sliders
        sliderMin.addEventListener('mousedown', function() {
            this.style.zIndex = '5';
            sliderMax.style.zIndex = '4';
        });

        sliderMax.addEventListener('mousedown', function() {
            this.style.zIndex = '5';
            sliderMin.style.zIndex = '4';
        });

        // Slider min
        sliderMin.addEventListener('input', function() {
            const min = parseInt(this.value);
            const max = parseInt(sliderMax.value);
            
            if (min > max - 1000000) { // Minimum 1M d'écart
                this.value = max - 1000000;
            }
            
            elements.priceMinInput.value = this.value;
            updateSliderRange();
            updatePriceDisplay();
        });

        // Slider max
        sliderMax.addEventListener('input', function() {
            const min = parseInt(sliderMin.value);
            const max = parseInt(this.value);
            
            if (max < min + 1000000) { // Minimum 1M d'écart
                this.value = min + 1000000;
            }
            
            elements.priceMaxInput.value = this.value;
            updateSliderRange();
            updatePriceDisplay();
        });

        // Synchroniser les inputs texte avec les sliders
        elements.priceMinInput.addEventListener('change', function() {
            const min = parseInt(this.value) || 0;
            const max = parseInt(elements.priceMaxInput.value) || CONFIG.maxPrice;
            
            if (min > max) {
                this.value = max;
            }
            
            sliderMin.value = this.value;
            updateSliderRange();
            updatePriceDisplay();
        });

        elements.priceMaxInput.addEventListener('change', function() {
            const min = parseInt(elements.priceMinInput.value) || 0;
            const max = parseInt(this.value) || CONFIG.maxPrice;
            
            if (max < min) {
                this.value = min;
            }
            
            sliderMax.value = this.value;
            updateSliderRange();
            updatePriceDisplay();
        });

        // Initialiser
        updateSliderRange();
    }

    /**
     * Mise à jour de l'affichage du prix
     */
    function updatePriceDisplay() {
        const min = parseInt(elements.priceMinInput.value) || 0;
        const max = parseInt(elements.priceMaxInput.value) || CONFIG.maxPrice;

        if (min === 0 && max === CONFIG.maxPrice) {
            elements.priceDisplay.textContent = 'Budget';
        } else if (min === 0) {
            elements.priceDisplay.textContent = `Jusqu'à ${formatPrice(max)}`;
        } else if (max === CONFIG.maxPrice) {
            elements.priceDisplay.textContent = `À partir de ${formatPrice(min)}`;
        } else {
            elements.priceDisplay.textContent = `${formatPrice(min)} - ${formatPrice(max)}`;
        }
    }

    /**
     * Formater le prix
     */
    function formatPrice(price) {
        if (price >= 1000000) {
            return (price / 1000000).toFixed(price % 1000000 === 0 ? 0 : 1) + 'M DA';
        } else if (price >= 1000) {
            return (price / 1000).toFixed(0) + 'K DA';
        }
        return price + ' DA';
    }

    /**
     * Configuration des boutons preset de prix
     */
    function setupPresetButtons() {
        const presetButtons = document.querySelectorAll('.preset-btn');
        const sliderMin = document.getElementById('price-slider-min');
        const sliderMax = document.getElementById('price-slider-max');
        const sliderRange = document.getElementById('price-slider-range');
        
        presetButtons.forEach(button => {
            button.addEventListener('click', function() {
                const min = parseInt(this.dataset.min) || 0;
                const max = parseInt(this.dataset.max) || CONFIG.maxPrice;

                elements.priceMinInput.value = min;
                elements.priceMaxInput.value = max;

                // Mettre à jour les sliders
                if (sliderMin && sliderMax) {
                    sliderMin.value = min;
                    sliderMax.value = max;
                    
                    // Mettre à jour la barre colorée
                    const percentMin = (min / CONFIG.maxPrice) * 100;
                    const percentMax = (max / CONFIG.maxPrice) * 100;
                    sliderRange.style.left = percentMin + '%';
                    sliderRange.style.right = (100 - percentMax) + '%';
                    
                    // Mettre à jour les labels
                    document.getElementById('slider-label-min').textContent = formatPrice(min);
                    document.getElementById('slider-label-max').textContent = formatPrice(max);
                }

                // Retirer active des autres boutons
                presetButtons.forEach(btn => btn.classList.remove('active'));
                this.classList.add('active');

                updatePriceDisplay();
            });
        });
    }

    /**
     * Gestion de la soumission du formulaire
     */
    function setupFormSubmit() {
        elements.searchForm.addEventListener('submit', function(e) {
            e.preventDefault();

            // Construire l'URL de recherche
            const params = new URLSearchParams();

            // Villes sélectionnées (envoyer toutes les villes)
            if (state.selectedCities.length > 0) {
                // Envoyer les IDs des villes comme array
                state.selectedCities.forEach(city => {
                    params.append('city_ids[]', city.id);
                });
                
                // Pour affichage: noms de toutes les villes sélectionnées
                const cityNames = state.selectedCities.map(c => c.label).join(',');
                params.append('cities', cityNames);
            }

            // Type de transaction
            const transaction = elements.transactionType.value;
            if (transaction) {
                params.append('transaction', transaction);
            }

            // Type de propriété
            const property = elements.propertyType.value;
            if (property) {
                params.append('property', property);
            }

            // Prix
            const priceMin = parseInt(elements.priceMinInput.value);
            const priceMax = parseInt(elements.priceMaxInput.value);
            
            if (priceMin > 0) {
                params.append('price_min', priceMin);
            }
            
            if (priceMax < CONFIG.maxPrice) {
                params.append('price_max', priceMax);
            }

            console.log('🔍 Recherche avec paramètres:', params.toString());

            // Redirection vers la page de résultats
            const searchUrl = `index.php?page=search&${params.toString()}`;
            window.location.href = searchUrl;
        });
    }

    /**
     * Initialisation au chargement du DOM
     */
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }

})();
