// ===== JavaScript pour taghtil =====

// Variables globales
let currentLanguage = 'fr';
let currentCurrency = 'DZD';
let currencyRates = {
    'DZD': 1,
    'EUR': 0.0074, // 1 DZD = 0.0074 EUR (approximatif)
    'USD': 0.0082  // 1 DZD = 0.0082 USD (approximatif)
};

// Traductions
const translations = {
    'fr': {
        'hero_title': 'Trouvez votre propriété idéale en Algérie',
        'hero_subtitle': 'Des milliers de biens immobiliers à vendre et à louer partout en Algérie',
        'transaction_type': 'Type de transaction',
        'property_type': 'Type de propriété',
        'city': 'Ville',
        'price_range': 'Gamme de prix',
        'search': 'Rechercher',
        'login': 'Connexion',
        'all_types': 'Tous types',
        'sale': 'Vente',
        'rent': 'Location',
        'all_properties': 'Toutes propriétés',
        'apartment': 'Appartement',
        'house': 'Maison',
        'villa': 'Villa',
        'land': 'Terrain',
        'commercial': 'Commercial',
        'all_cities': 'Toutes les villes',
        'min': 'Min',
        'max': 'Max',
        'property_types_title': 'Types de Propriétés',
        'property_types_subtitle': 'Découvrez nos différents types de biens immobiliers',
        'apartments': 'Appartements',
        'houses': 'Maisons',
        'villas': 'Villas',
        'lands': 'Terrains',
        'offices': 'Bureaux',
        'apartments_desc': 'Modernes et confortables en centre-ville',
        'houses_desc': 'Familiales avec jardin et garage',
        'villas_desc': 'Luxueuses avec piscine et grand jardin',
        'lands_desc': 'Constructibles dans toute l\'Algérie',
        'commercial_desc': 'Locaux commerciaux et bureaux',
        'offices_desc': 'Espaces professionnels équipés',
        'view_all': 'Voir tout',
        'nearby_properties': 'Propriétés à Proximité',
        'nearby_properties_subtitle': 'Découvrez les biens immobiliers près de chez vous',
        'load_more': 'Voir plus de propriétés',
        'about_us': 'À Propos',
        'about_description': 'taghtil est la plateforme immobilière de référence en Algérie. Nous facilitons l\'achat, la vente et la location de biens immobiliers.',
        'quick_links': 'Liens Rapides',
        'contact': 'Contact',
        'address': 'Alger, Algérie',
        'hours': 'Lun - Ven: 8h - 18h',
        'copyright': '© 2024 taghtil. Tous droits réservés.',
        'privacy': 'Confidentialité',
        'terms': 'Conditions',
        'sitemap': 'Plan du site',
        'total_properties': 'Propriétés disponibles',
        'for_sale': 'À vendre',
        'for_rent': 'À louer',
        'cities_covered': 'Villes couvertes',
        'search_properties': 'Rechercher des propriétés',
        'post_ad': 'Déposer une annonce',
        'become_agent': 'Devenir agent',
        'price_estimate': 'Estimation gratuite',
        'help': 'Aide',
        'apartments_for_sale': 'Appartements à vendre',
        'houses_for_sale': 'Maisons à vendre',
        'villas_for_sale': 'Villas à vendre',
        'apartments_for_rent': 'Appartements à louer',
        'commercial_properties': 'Biens commerciaux',
        'visitor_space': 'Espace Visiteur',
        'admin_space': 'Espace Administrateur',
        'auto_login': 'Connexion Automatique'
    },
    'en': {
        'hero_title': 'Find your ideal property in Algeria',
        'hero_subtitle': 'Thousands of real estate properties for sale and rent throughout Algeria',
        'transaction_type': 'Transaction type',
        'property_type': 'Property type',
        'city': 'City',
        'price_range': 'Price range',
        'search': 'Search',
        'login': 'Login',
        'all_types': 'All types',
        'sale': 'Sale',
        'rent': 'Rent',
        'all_properties': 'All properties',
        'apartment': 'Apartment',
        'house': 'House',
        'villa': 'Villa',
        'land': 'Land',
        'commercial': 'Commercial',
        'all_cities': 'All cities',
        'min': 'Min',
        'max': 'Max',
        'property_types_title': 'Property Types',
        'property_types_subtitle': 'Discover our different types of real estate',
        'apartments': 'Apartments',
        'houses': 'Houses',
        'villas': 'Villas',
        'lands': 'Lands',
        'offices': 'Offices',
        'apartments_desc': 'Modern and comfortable in city center',
        'houses_desc': 'Family homes with garden and garage',
        'villas_desc': 'Luxurious with pool and large garden',
        'lands_desc': 'Buildable throughout Algeria',
        'commercial_desc': 'Commercial premises and offices',
        'offices_desc': 'Equipped professional spaces',
        'view_all': 'View all',
        'nearby_properties': 'Nearby Properties',
        'nearby_properties_subtitle': 'Discover real estate near you',
        'load_more': 'Load more properties',
        'about_us': 'About Us',
        'about_description': 'taghtil is the reference real estate platform in Algeria. We facilitate buying, selling and renting real estate.',
        'quick_links': 'Quick Links',
        'contact': 'Contact',
        'address': 'Algiers, Algeria',
        'hours': 'Mon - Fri: 8am - 6pm',
        'copyright': '© 2024 taghtil. All rights reserved.',
        'privacy': 'Privacy',
        'terms': 'Terms',
        'sitemap': 'Sitemap',
        'total_properties': 'Available properties',
        'for_sale': 'For sale',
        'for_rent': 'For rent',
        'cities_covered': 'Cities covered',
        'search_properties': 'Search properties',
        'post_ad': 'Post an ad',
        'become_agent': 'Become an agent',
        'price_estimate': 'Free estimate',
        'help': 'Help',
        'apartments_for_sale': 'Apartments for sale',
        'houses_for_sale': 'Houses for sale',
        'villas_for_sale': 'Villas for sale',
        'apartments_for_rent': 'Apartments for rent',
        'commercial_properties': 'Commercial properties',
        'visitor_space': 'Visitor Space',
        'admin_space': 'Administrator Space',
        'auto_login': 'Auto Login'
    }
};

// ===== Initialisation =====
document.addEventListener('DOMContentLoaded', function() {
    // Initialiser les fonctionnalités
    initializeLanguage();
    initializeCurrency();
    initializeSearch();
    initializeAnimations();
    
    // Mettre à jour les statistiques (seulement si les éléments de stats existent)
    if (document.getElementById('stat-total') || document.querySelector('.stat-number')) {
        updateStatistics();
    }
    
    // Event listeners avec vérification d'existence
    // Désactivé : Le formulaire soumet maintenant directement vers la page de recherche
    // const searchForm = document.getElementById('search-form');
    // if (searchForm) {
    //     searchForm.addEventListener('submit', handleSearch);
    // }
    
    // Géolocalisation pour les propriétés à proximité
    if (navigator.geolocation) {
        navigator.geolocation.getCurrentPosition(
            function(position) {
                console.log('Position détectée:', position.coords.latitude, position.coords.longitude);
                // Pour l'instant, on garde les propriétés par défaut chargées par PHP
            },
            function(error) {
                console.log('Géolocalisation non disponible:', error);
                // Les propriétés sont déjà chargées par PHP
            }
        );
    }
});

// ===== Fonctions de langue =====
function changeLanguage(lang) {
    currentLanguage = lang;
    
    // Vérifier si l'élément existe avant de l'utiliser
    const currentLangElement = document.getElementById('current-lang');
    if (currentLangElement) {
        currentLangElement.textContent = lang.toUpperCase();
    }
    
    // Mettre à jour tous les éléments avec data-translate
    document.querySelectorAll('[data-translate]').forEach(element => {
        const key = element.getAttribute('data-translate');
        if (translations[lang] && translations[lang][key]) {
            if (element.tagName === 'INPUT' && element.type === 'text') {
                element.placeholder = translations[lang][key];
            } else {
                element.textContent = translations[lang][key];
            }
        }
    });
    
    // Mettre à jour les placeholders
    document.querySelectorAll('[data-translate-placeholder]').forEach(element => {
        const key = element.getAttribute('data-translate-placeholder');
        if (translations[lang] && translations[lang][key]) {
            element.placeholder = translations[lang][key];
        }
    });
    
    // Sauvegarder la préférence
    localStorage.setItem('language', lang);
}

function initializeLanguage() {
    const savedLang = localStorage.getItem('language') || 'fr';
    changeLanguage(savedLang);
}

// ===== Fonctions de monnaie =====
function changeCurrency(currency, flag) {
    currentCurrency = currency;
    
    // Vérifier si l'élément existe avant de l'utiliser
    const currentCurrencyElement = document.getElementById('current-currency');
    if (currentCurrencyElement) {
        currentCurrencyElement.textContent = currency;
    }
    
    // Mettre à jour tous les prix affichés
    updatePricesDisplay();
    
    // Sauvegarder la préférence
    localStorage.setItem('currency', currency);
}

function initializeCurrency() {
    const savedCurrency = localStorage.getItem('currency') || 'DZD';
    changeCurrency(savedCurrency);
}

function convertPrice(price, fromCurrency = 'DZD', toCurrency = currentCurrency) {
    if (fromCurrency === toCurrency) return price;
    
    // Convertir d'abord en DZD si nécessaire
    let priceInDZD = price;
    if (fromCurrency !== 'DZD') {
        priceInDZD = price / currencyRates[fromCurrency];
    }
    
    // Puis convertir vers la monnaie cible
    return Math.round(priceInDZD * currencyRates[toCurrency]);
}

function formatPrice(price, currency = currentCurrency) {
    const formatters = {
        'DZD': new Intl.NumberFormat('fr-DZ', { style: 'currency', currency: 'DZD' }),
        'EUR': new Intl.NumberFormat('fr-FR', { style: 'currency', currency: 'EUR' }),
        'USD': new Intl.NumberFormat('en-US', { style: 'currency', currency: 'USD' })
    };
    
    return formatters[currency] ? formatters[currency].format(price) : `${price} ${currency}`;
}

function updatePricesDisplay() {
    document.querySelectorAll('[data-price]').forEach(element => {
        const originalPrice = parseInt(element.getAttribute('data-price'));
        const convertedPrice = convertPrice(originalPrice);
        element.textContent = formatPrice(convertedPrice);
    });
}

// ===== Fonctions de recherche =====
function initializeSearch() {
    // Auto-complétion pour les villes
    const villeSelect = document.getElementById('ville');
    // Vous pouvez ajouter plus de villes depuis votre base de données
}

function handleSearch(event) {
    event.preventDefault();
    
    const formData = new FormData(event.target);
    const searchParams = new URLSearchParams({
        action: 'search',
        transaction_type: formData.get('transaction_type') || '',
        property_type: formData.get('property_type') || '',
        ville: formData.get('ville') || '',
        prix_min: formData.get('prix_min') || '',
        prix_max: formData.get('prix_max') || ''
    });
    
    // Afficher un indicateur de chargement
    showLoading();
    
    // Effectuer la recherche via API
    fetch(`api_public.php?${searchParams}`)
        .then(response => response.json())
        .then(data => {
            hideLoading();
            if (data.success) {
                displaySearchResults(data.data);
            } else {
                alert('Erreur lors de la recherche');
            }
        })
        .catch(error => {
            hideLoading();
            console.error('Erreur:', error);
            alert('Erreur de connexion');
        });
}

function displaySearchResults(properties) {
    const container = document.getElementById('nearby-properties');
    container.innerHTML = '';
    
    if (properties.length === 0) {
        container.innerHTML = `
            <div class="col-12 text-center">
                <div class="alert alert-info">
                    <i class="fas fa-search me-2"></i>
                    Aucune propriété trouvée pour ces critères.
                </div>
            </div>
        `;
        return;
    }
    
    properties.forEach(property => {
        const propertyCard = createPropertyCardFromDB(property);
        container.appendChild(propertyCard);
    });
    
    // Scroll vers les résultats
    container.scrollIntoView({ behavior: 'smooth' });
}

function createPropertyCardFromDB(property) {
    const col = document.createElement('div');
    col.className = 'col-lg-4 col-md-6';
    
    const convertedPrice = convertPrice(property.prix);
    const formattedPrice = formatPrice(convertedPrice);
    
    // Déterminer l'image selon la catégorie
    const imageUrl = getImageByCategory(property.categorie_nom);
    
    col.innerHTML = `
        <div class="property-card">
            <div class="property-image">
                <img src="${imageUrl}" alt="${property.titre}" loading="lazy">
                <div class="property-badge">${property.type_transaction || 'Vente'}</div>
                <div class="property-price" data-price="${property.prix}">${formattedPrice}</div>
            </div>
            <div class="property-info">
                <h5 class="property-title">${property.titre.length > 50 ? property.titre.substring(0, 50) + '...' : property.titre}</h5>
                <div class="property-location">
                    <i class="fas fa-map-marker-alt me-1"></i>
                    ${property.ville}
                </div>
                <div class="property-features">
                    <div class="feature">
                        <i class="fas fa-expand-arrows-alt"></i>
                        <span>${property.surface} m²</span>
                    </div>
                    ${property.nb_pieces > 0 ? `
                        <div class="feature">
                            <i class="fas fa-bed"></i>
                            <span>${property.nb_pieces} pièces</span>
                        </div>
                    ` : ''}
                    ${property.nb_chambres > 0 ? `
                        <div class="feature">
                            <i class="fas fa-door-open"></i>
                            <span>${property.nb_chambres} ch.</span>
                        </div>
                    ` : ''}
                </div>
                <div class="property-actions">
                    <button class="btn btn-primary btn-sm" onclick="voirDetails(${property.id})">
                        <i class="fas fa-eye me-1"></i>
                        Voir détails
                    </button>
                    <button class="btn btn-outline-primary btn-sm" onclick="ajouterFavoris(${property.id})">
                        <i class="fas fa-heart me-1"></i>
                        Favoris
                    </button>
                </div>
            </div>
        </div>
    `;
    
    return col;
}

function getImageByCategory(category) {
    const images = {
        'appartement': 'https://images.unsplash.com/photo-1560448204-e02f11c3d0e2?w=400',
        'maison': 'https://images.unsplash.com/photo-1570129477492-45c003edd2be?w=400',
        'villa': 'https://images.unsplash.com/photo-1564013799919-ab600027ffc6?w=400',
        'terrain': 'https://images.unsplash.com/photo-1500382017468-9049fed747ef?w=400',
        'commercial': 'https://images.unsplash.com/photo-1497366216548-37526070297c?w=400',
        'bureau': 'https://images.unsplash.com/photo-1497366811353-6870744d04b2?w=400'
    };
    
    const categoryLower = (category || '').toLowerCase();
    return images[categoryLower] || 'https://images.unsplash.com/photo-1560518883-ce09059eeffa?w=400';
}

// ===== Chargement des propriétés à proximité =====
function loadNearbyProperties() {
    const container = document.getElementById('nearby-properties');
    
    // Afficher un indicateur de chargement
    container.innerHTML = '<div class="col-12 text-center"><div class="spinner"></div></div>';
    
    // Simuler le chargement des propriétés
    setTimeout(() => {
        const properties = generateSampleProperties();
        displayProperties(properties, container);
    }, 1500);
}

function loadPropertiesByLocation(lat, lng) {
    // Cette fonction utiliserait la géolocalisation pour charger les propriétés
    console.log('Chargement des propriétés pour:', lat, lng);
    loadNearbyProperties(); // Pour l'instant, charger les propriétés par défaut
}

function generateSampleProperties() {
    const sampleProperties = [
        {
            id: 1,
            title: 'Appartement moderne 3 pièces',
            price: 15000000,
            location: 'Hydra, Alger',
            image: 'https://images.unsplash.com/photo-1560448204-e02f11c3d0e2?w=400',
            type: 'Vente',
            surface: 85,
            rooms: 3,
            bathrooms: 2
        },
        {
            id: 2,
            title: 'Villa avec piscine',
            price: 45000000,
            location: 'Bouzareah, Alger',
            image: 'https://images.unsplash.com/photo-1564013799919-ab600027ffc6?w=400',
            type: 'Vente',
            surface: 250,
            rooms: 5,
            bathrooms: 3
        },
        {
            id: 3,
            title: 'Maison familiale',
            price: 25000000,
            location: 'Bab Ezzouar, Alger',
            image: 'https://images.unsplash.com/photo-1570129477492-45c003edd2be?w=400',
            type: 'Vente',
            surface: 150,
            rooms: 4,
            bathrooms: 2
        },
        {
            id: 4,
            title: 'Appartement à louer F3',
            price: 45000,
            location: 'Didouche Mourad, Alger',
            image: 'https://images.unsplash.com/photo-1502672260266-1c1ef2d93688?w=400',
            type: 'Location',
            surface: 75,
            rooms: 3,
            bathrooms: 1
        },
        {
            id: 5,
            title: 'Studio meublé centre-ville',
            price: 25000,
            location: 'Centre-ville, Alger',
            image: 'https://images.unsplash.com/photo-1522708323590-d24dbb6b0267?w=400',
            type: 'Location',
            surface: 35,
            rooms: 1,
            bathrooms: 1
        },
        {
            id: 6,
            title: 'Terrain constructible',
            price: 8000000,
            location: 'Douera, Alger',
            image: 'https://images.unsplash.com/photo-1500382017468-9049fed747ef?w=400',
            type: 'Vente',
            surface: 400,
            rooms: 0,
            bathrooms: 0
        }
    ];
    
    return sampleProperties;
}

function displayProperties(properties, container) {
    container.innerHTML = '';
    
    properties.forEach(property => {
        const propertyCard = createPropertyCard(property);
        container.appendChild(propertyCard);
    });
}

function createPropertyCard(property) {
    const col = document.createElement('div');
    col.className = 'col-lg-4 col-md-6';
    
    const convertedPrice = convertPrice(property.price);
    const formattedPrice = formatPrice(convertedPrice);
    
    col.innerHTML = `
        <div class="property-card">
            <div class="property-image">
                <img src="${property.image}" alt="${property.title}" loading="lazy">
                <div class="property-badge">${property.type}</div>
                <div class="property-price" data-price="${property.price}">${formattedPrice}</div>
            </div>
            <div class="property-info">
                <h5 class="property-title">${property.title}</h5>
                <div class="property-location">
                    <i class="fas fa-map-marker-alt me-1"></i>
                    ${property.location}
                </div>
                <div class="property-features">
                    <div class="feature">
                        <i class="fas fa-expand-arrows-alt"></i>
                        <span>${property.surface} m²</span>
                    </div>
                    ${property.rooms > 0 ? `
                        <div class="feature">
                            <i class="fas fa-bed"></i>
                            <span>${property.rooms} pièces</span>
                        </div>
                    ` : ''}
                    ${property.bathrooms > 0 ? `
                        <div class="feature">
                            <i class="fas fa-bath"></i>
                            <span>${property.bathrooms} sdb</span>
                        </div>
                    ` : ''}
                </div>
                <div class="property-actions">
                    <button class="btn btn-primary btn-sm">
                        <i class="fas fa-eye me-1"></i>
                        Voir détails
                    </button>
                    <button class="btn btn-outline-primary btn-sm">
                        <i class="fas fa-heart me-1"></i>
                        Favoris
                    </button>
                </div>
            </div>
        </div>
    `;
    
    return col;
}

function loadMoreProperties() {
    const container = document.getElementById('nearby-properties');
    
    // Afficher un indicateur de chargement
    const loadingDiv = document.createElement('div');
    loadingDiv.className = 'col-12 text-center';
    loadingDiv.innerHTML = '<div class="spinner"></div>';
    container.appendChild(loadingDiv);
    
    // Détecter le bon chemin pour l'API selon l'emplacement de la page
    const apiPath = window.location.pathname.includes('/visiteur/') ? '../api_public.php' : 'api_public.php';
    
    // Charger plus de propriétés via API
    fetch(apiPath + '?action=get_nearby')
        .then(response => response.json())
        .then(data => {
            // Supprimer l'indicateur de chargement
            loadingDiv.remove();
            
            if (data.success && data.data.length > 0) {
                data.data.forEach(property => {
                    const propertyCard = createPropertyCardFromDB(property);
                    container.appendChild(propertyCard);
                });
            } else {
                const noMoreDiv = document.createElement('div');
                noMoreDiv.className = 'col-12 text-center';
                noMoreDiv.innerHTML = `
                    <div class="alert alert-info">
                        <i class="fas fa-info-circle me-2"></i>
                        Aucune autre propriété à afficher.
                    </div>
                `;
                container.appendChild(noMoreDiv);
            }
        })
        .catch(error => {
            loadingDiv.remove();
            console.error('Erreur:', error);
        });
}

// Nouvelles fonctions pour les actions des propriétés
function voirDetails(propertyId) {
    // Rediriger vers la page de détails de la propriété
    window.location.href = `property_details.php?id=${propertyId}`;
}

function ajouterFavoris(propertyId) {
    // Ajouter à la liste des favoris (nécessite une session utilisateur)
    alert(`Propriété ${propertyId} ajoutée aux favoris ! (Fonctionnalité à développer)`);
}

// Fonction pour mettre à jour les statistiques en temps réel
function updateStatistics() {
    // Détecter le bon chemin pour l'API selon l'emplacement de la page
    const apiPath = window.location.pathname.includes('/visiteur/') ? '../api_public.php' : 'api_public.php';
    
    fetch(apiPath + '?action=get_stats')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const stats = data.data;
                
                // Vérifier si les éléments existent avant de les mettre à jour
                const statTotalElement = document.getElementById('stat-total');
                if (statTotalElement) statTotalElement.textContent = stats.total_annonces;
                
                const statVentesElement = document.getElementById('stat-ventes');
                if (statVentesElement) statVentesElement.textContent = stats.total_ventes;
                
                const statLocationsElement = document.getElementById('stat-locations');
                if (statLocationsElement) statLocationsElement.textContent = stats.total_locations;
                
                const statVillesElement = document.getElementById('stat-villes');
                if (statVillesElement) statVillesElement.textContent = stats.total_villes;
            }
        })
        .catch(error => {
            console.error('Erreur lors de la mise à jour des statistiques:', error);
        });
}

// ===== Animations =====
function initializeAnimations() {
    // Observer pour les animations au scroll
    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('animate-on-scroll');
            }
        });
    });
    
    // Observer les cartes de propriétés
    document.querySelectorAll('.property-type-card, .property-card').forEach(card => {
        observer.observe(card);
    });
}

// ===== Utilitaires =====
function showLoading() {
    // Afficher un indicateur de chargement global
    const loading = document.createElement('div');
    loading.id = 'global-loading';
    loading.innerHTML = `
        <div class="position-fixed top-0 start-0 w-100 h-100 d-flex align-items-center justify-content-center" 
             style="background: rgba(0,0,0,0.5); z-index: 9999;">
            <div class="spinner"></div>
        </div>
    `;
    document.body.appendChild(loading);
}

function hideLoading() {
    const loading = document.getElementById('global-loading');
    if (loading) {
        loading.remove();
    }
}

// ===== Navigation fluide =====
document.querySelectorAll('a[href^="#"]').forEach(anchor => {
    anchor.addEventListener('click', function (e) {
        e.preventDefault();
        const target = document.querySelector(this.getAttribute('href'));
        if (target) {
            target.scrollIntoView({
                behavior: 'smooth',
                block: 'start'
            });
        }
    });
});

// ===== Gestion des erreurs =====
window.addEventListener('error', function(e) {
    console.error('Erreur JavaScript:', e.error);
});

// ===== Export pour utilisation externe =====
window.taghtil = {
    changeLanguage,
    changeCurrency,
    // searchProperties, // TODO: À implémenter
    loadMoreProperties
};