<?php
// ===== Configuration et initialisation =====
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Inclusion des classes nécessaires
include_once 'Parametre/taghilt/taghilt.php';
include_once 'Parametre/taghilt/LoadDirClass.php';
include_once 'Parametre/taghilt/dbConnect.php';

// Connexion à la base de données
try {
    $conn = dbConnect(false);
} catch (Exception $e) {
    error_log("Erreur de connexion DB: " . $e->getMessage());
    $conn = null;
}

// ===== Fonctions pour récupérer les données =====

/**
 * Récupère les villes avec des annonces actives
 */
function getVillesAvecAnnonces($conn) {
    if (!$conn) return [];
    
    try {
        $sql = "SELECT DISTINCT ville 
                FROM annonces 
                WHERE actif = 1 AND ville IS NOT NULL AND ville != '' 
                ORDER BY ville";
        $conn->exec($sql);
        return $conn->data;
    } catch (Exception $e) {
        error_log("Erreur getVillesAvecAnnonces: " . $e->getMessage());
        return [];
    }
}

/**
 * Récupère les types de propriétés disponibles
 */
function getTypesProprietes($conn) {
    if (!$conn) return [];
    
    try {
        $sql = "SELECT id, nom, description 
                FROM categories_biens 
                WHERE actif = 1 
                ORDER BY nom";
        $conn->exec($sql);
        return $conn->data;
    } catch (Exception $e) {
        error_log("Erreur getTypesProprietes: " . $e->getMessage());
        return [];
    }
}

/**
 * Récupère les propriétés selon les critères
 */
function rechercherProprietes($conn, $criteres = []) {
    if (!$conn) return [];
    
    try {
        $whereConditions = ["a.actif = 1", "u.actif = 1"];
        $params = [];
        
        // Filtre par type de transaction
        if (!empty($criteres['transaction_type'])) {
            $whereConditions[] = "tt.nom = ?";
            $params[] = $criteres['transaction_type'];
        }
        
        // Filtre par ville
        if (!empty($criteres['ville'])) {
            $whereConditions[] = "a.ville = ?";
            $params[] = $criteres['ville'];
        }
        
        // Filtre par type de propriété
        if (!empty($criteres['property_type'])) {
            $whereConditions[] = "cb.nom = ?";
            $params[] = $criteres['property_type'];
        }
        
        // Filtre par prix minimum
        if (!empty($criteres['prix_min']) && is_numeric($criteres['prix_min'])) {
            $whereConditions[] = "a.prix >= ?";
            $params[] = (float)$criteres['prix_min'];
        }
        
        // Filtre par prix maximum
        if (!empty($criteres['prix_max']) && is_numeric($criteres['prix_max'])) {
            $whereConditions[] = "a.prix <= ?";
            $params[] = (float)$criteres['prix_max'];
        }
        
        $whereClause = implode(' AND ', $whereConditions);
        
        $sql = "SELECT 
                    a.id, a.titre, a.description, a.prix, a.surface, 
                    a.nb_pieces, a.nb_chambres, a.ville, a.adresse,
                    a.statut, a.date_creation, a.date_publication, 
                    a.vues, a.contacts, a.mise_en_avant,
                    u.id as user_id, u.username, u.prenom, u.nom,
                    CONCAT(u.prenom, ' ', u.nom) AS proprietaire,
                    cb.nom as categorie_nom,
                    tt.nom as type_transaction
                FROM annonces a
                LEFT JOIN users u ON a.user_id = u.id
                LEFT JOIN categories_biens cb ON a.categorie_id = cb.id
                LEFT JOIN types_transaction tt ON a.type_transaction_id = tt.id
                WHERE $whereClause
                ORDER BY a.mise_en_avant DESC, a.date_creation DESC
                LIMIT 12";
        
        $conn->exec($sql);
        return $conn->data;
    } catch (Exception $e) {
        error_log("Erreur rechercherProprietes: " . $e->getMessage());
        return [];
    }
}

/**
 * Récupère les propriétés en vedette (mise en avant)
 */
function getProprietesVedette($conn, $limit = 6) {
    if (!$conn) return [];
    
    try {
        $sql = "SELECT 
                    a.id, a.titre, a.description, a.prix, a.surface, 
                    a.nb_pieces, a.nb_chambres, a.ville, a.adresse,
                    a.statut, a.date_creation, a.vues, a.contacts,
                    CONCAT(u.prenom, ' ', u.nom) AS proprietaire,
                    cb.nom as categorie_nom,
                    tt.nom as type_transaction
                FROM annonces a
                LEFT JOIN users u ON a.user_id = u.id
                LEFT JOIN categories_biens cb ON a.categorie_id = cb.id
                LEFT JOIN types_transaction tt ON a.type_transaction_id = tt.id
                WHERE a.actif = 1 AND u.actif = 1 AND a.mise_en_avant = 1
                ORDER BY a.date_creation DESC
                LIMIT ?";
        
        $conn->exec($sql);
        return $conn->data;
    } catch (Exception $e) {
        error_log("Erreur getProprietesVedette: " . $e->getMessage());
        return [];
    }
}

/**
 * Récupère les statistiques pour la page d'accueil
 */
function getStatistiquesAccueil($conn) {
    if (!$conn) {
        return [
            'total_annonces' => 0,
            'total_ventes' => 0,
            'total_locations' => 0,
            'total_villes' => 0
        ];
    }
    
    try {
        // Total des annonces actives
        $conn->exec("SELECT COUNT(*) as count FROM annonces WHERE actif = 1");
        $total_annonces = $conn->data[0]['count'];
        
        // Annonces de vente
        $conn->exec("SELECT COUNT(*) as count FROM annonces a 
                     LEFT JOIN types_transaction tt ON a.type_transaction_id = tt.id 
                     WHERE a.actif = 1 AND tt.nom = 'vente'");
        $total_ventes = $conn->data[0]['count'];
        
        // Annonces de location
        $conn->exec("SELECT COUNT(*) as count FROM annonces a 
                     LEFT JOIN types_transaction tt ON a.type_transaction_id = tt.id 
                     WHERE a.actif = 1 AND tt.nom = 'location'");
        $total_locations = $conn->data[0]['count'];
        
        // Nombre de villes
        $conn->exec("SELECT COUNT(DISTINCT ville) as count FROM annonces WHERE actif = 1 AND ville IS NOT NULL");
        $total_villes = $conn->data[0]['count'];
        
        return [
            'total_annonces' => $total_annonces,
            'total_ventes' => $total_ventes,
            'total_locations' => $total_locations,
            'total_villes' => $total_villes
        ];
    } catch (Exception $e) {
        error_log("Erreur getStatistiquesAccueil: " . $e->getMessage());
        return [
            'total_annonces' => 0,
            'total_ventes' => 0,
            'total_locations' => 0,
            'total_villes' => 0
        ];
    }
}

// ===== Traitement des requêtes AJAX =====
if (isset($_GET['action']) && $_GET['action'] === 'search') {
    header('Content-Type: application/json');
    
    $criteres = [
        'transaction_type' => $_GET['transaction_type'] ?? '',
        'property_type' => $_GET['property_type'] ?? '',
        'ville' => $_GET['ville'] ?? '',
        'prix_min' => $_GET['prix_min'] ?? '',
        'prix_max' => $_GET['prix_max'] ?? ''
    ];
    
    $proprietes = rechercherProprietes($conn, $criteres);
    
    echo json_encode([
        'success' => true,
        'data' => $proprietes,
        'count' => count($proprietes)
    ]);
    exit;
}

if (isset($_GET['action']) && $_GET['action'] === 'get_nearby') {
    header('Content-Type: application/json');
    
    $proprietes = getProprietesVedette($conn, 12);
    
    echo json_encode([
        'success' => true,
        'data' => $proprietes
    ]);
    exit;
}

if (isset($_GET['action']) && $_GET['action'] === 'get_stats') {
    header('Content-Type: application/json');
    
    $stats = getStatistiquesAccueil($conn);
    
    echo json_encode([
        'success' => true,
        'data' => $stats
    ]);
    exit;
}

/**
 * Récupère les annonces récentes
 */
function getAnnoncesRecentes($conn, $limit = 6) {
    if (!$conn) return [];
    
    try {
        $sql = "SELECT 
                    a.id,
                    a.titre,
                    a.prix,
                    a.surface,
                    a.chambres,
                    a.salles_bain,
                    a.ville,
                    a.type_transaction,
                    a.date_creation,
                    tp.nom as type_propriete
                FROM annonces a
                LEFT JOIN types_proprietes tp ON a.type_propriete_id = tp.id
                WHERE a.actif = 1 
                ORDER BY a.date_creation DESC
                LIMIT " . (int)$limit;
        
        $conn->exec($sql);
        return $conn->data;
        
    } catch (Exception $e) {
        error_log("Erreur lors de la récupération des annonces récentes: " . $e->getMessage());
        return [];
    }
}

// ===== Chargement des données pour la page =====
$villes = getVillesAvecAnnonces($conn);
$typesProprietes = getTypesProprietes($conn);
$proprietesVedette = getProprietesVedette($conn);
$statistiques = getStatistiquesAccueil($conn);

// Ajouter les images par défaut selon le type
function getImageParDefaut($categorie) {
    $images = [
        'appartement' => 'https://images.unsplash.com/photo-1560448204-e02f11c3d0e2?w=400',
        'maison' => 'https://images.unsplash.com/photo-1570129477492-45c003edd2be?w=400',
        'villa' => 'https://images.unsplash.com/photo-1564013799919-ab600027ffc6?w=400',
        'terrain' => 'https://images.unsplash.com/photo-1500382017468-9049fed747ef?w=400',
        'commercial' => 'https://images.unsplash.com/photo-1497366216548-37526070297c?w=400',
        'bureau' => 'https://images.unsplash.com/photo-1497366811353-6870744d04b2?w=400'
    ];
    
    $categorie_lower = strtolower($categorie);
    return $images[$categorie_lower] ?? 'https://images.unsplash.com/photo-1560518883-ce09059eeffa?w=400';
}
?>