<?php
/**
 * API pour la recherche de localités algériennes
 * Utilisé pour l'autocomplétion dans le formulaire de recherche
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');

require_once __DIR__ . '/config.php';

try {
    $pdo = new PDO(
        "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4",
        DB_USER,
        DB_PASS,
        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
    );
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Erreur de connexion à la base de données']);
    exit;
}

$action = $_GET['action'] ?? 'search';

switch ($action) {
    case 'search':
        searchLocalites($pdo);
        break;
    
    case 'wilayas':
        getWilayas($pdo);
        break;
    
    case 'communes':
        getCommunes($pdo);
        break;
    
    default:
        http_response_code(400);
        echo json_encode(['error' => 'Action non reconnue']);
}

/**
 * Recherche des localités par nom (autocomplétion)
 */
function searchLocalites($pdo) {
    $query = $_GET['q'] ?? '';
    
    if (strlen($query) < 2) {
        echo json_encode([]);
        return;
    }
    
    // Recherche dans les noms avec priorité : communes > wilayas
    $stmt = $pdo->prepare("
        SELECT 
            id,
            nom,
            nom_ar,
            type,
            code_wilaya,
            code_postal,
            CASE 
                WHEN nom LIKE :exact THEN 1
                WHEN nom LIKE :start THEN 2
                WHEN type = 'commune' THEN 3
                WHEN type = 'wilaya' THEN 4
                ELSE 5
            END as priority
        FROM localites_algerie
        WHERE 
            nom LIKE :search 
            OR nom_ar LIKE :search
        ORDER BY priority ASC, nom ASC
        LIMIT 20
    ");
    
    $searchTerm = '%' . $query . '%';
    $exactTerm = $query;
    $startTerm = $query . '%';
    
    $stmt->execute([
        'search' => $searchTerm,
        'exact' => $exactTerm,
        'start' => $startTerm
    ]);
    
    $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Formater les résultats pour l'autocomplétion
    $formatted = array_map(function($row) use ($pdo) {
        $label = $row['nom'];
        
        // Si c'est une commune, ajouter la wilaya
        if ($row['type'] === 'commune') {
            $wilayaStmt = $pdo->prepare("
                SELECT nom FROM localites_algerie 
                WHERE code_wilaya = :code AND type = 'wilaya'
            ");
            $wilayaStmt->execute(['code' => $row['code_wilaya']]);
            $wilaya = $wilayaStmt->fetch(PDO::FETCH_ASSOC);
            
            if ($wilaya) {
                $label .= ', ' . $wilaya['nom'];
            }
        }
        
        return [
            'id' => $row['id'],
            'value' => $row['nom'],
            'label' => $label,
            'type' => $row['type'],
            'code_wilaya' => $row['code_wilaya'],
            'code_postal' => $row['code_postal']
        ];
    }, $results);
    
    echo json_encode($formatted);
}

/**
 * Récupérer toutes les wilayas
 */
function getWilayas($pdo) {
    $stmt = $pdo->query("
        SELECT id, nom, nom_ar, code_wilaya, latitude, longitude
        FROM localites_algerie
        WHERE type = 'wilaya'
        ORDER BY CAST(code_wilaya AS UNSIGNED) ASC
    ");
    
    $wilayas = $stmt->fetchAll(PDO::FETCH_ASSOC);
    echo json_encode($wilayas);
}

/**
 * Récupérer les communes d'une wilaya
 */
function getCommunes($pdo) {
    $codeWilaya = $_GET['wilaya'] ?? '';
    
    if (empty($codeWilaya)) {
        echo json_encode([]);
        return;
    }
    
    $stmt = $pdo->prepare("
        SELECT id, nom, nom_ar, code_postal
        FROM localites_algerie
        WHERE type = 'commune' AND code_wilaya = :code
        ORDER BY nom ASC
    ");
    
    $stmt->execute(['code' => $codeWilaya]);
    $communes = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo json_encode($communes);
}
