<?php

namespace Classe;

use PDO;

class PublicIndex
{
    private $typesProprietes;
    private $wilayasAlgerie;
    private $villesAvecAnnonces;
    private $annoncesRecentes;
    private $pathPrefix;
    
    public function __construct()
    {
        $this->pathPrefix = $this->detectPathPrefix();
        $this->initializeData();
    }
    
    private function detectPathPrefix()
    {
        // Détecter le préfixe de chemin selon le contexte
        if (strpos($_SERVER['REQUEST_URI'], '/admin/') !== false || 
            strpos($_SERVER['REQUEST_URI'], '/visiteur/') !== false) {
            return '../';
        }
        return '';
    }
    
    private function initializeData()
    {
        // Inclusion de l'API pour récupérer les vraies données
        include_once 'api_public.php';
        
        try {
            // Connexion à la base de données
            include_once 'Parametre/taghilt/taghilt.php';
            include_once 'Parametre/taghilt/LoadDirClass.php';
            include_once 'Parametre/taghilt/dbConnect.php';
            
            $conn = dbConnect(true); // Activer le mode verbose
            
            if (!$conn) {
                throw new \Exception('Impossible de se connecter à la base de données');
            }
            
            // Récupérer les vraies données depuis la base
            $this->typesProprietes = getTypesProprietes($conn);
            $this->villesAvecAnnonces = getVillesAvecAnnonces($conn);
            
            // Récupérer les annonces récentes
            if (function_exists('getAnnoncesRecentes')) {
                $this->annoncesRecentes = getAnnoncesRecentes($conn, 6);
            } else {
                $this->annoncesRecentes = $this->getAnnoncesRecentesManual($conn);
            }
            
        } catch (\Exception $e) {
            error_log("Erreur lors de l'initialisation des données: " . $e->getMessage());
            
            // Données par défaut en cas d'erreur
            $this->typesProprietes = [
                ['id' => 1, 'nom' => 'Appartement', 'description' => 'Appartements et studios'],
                ['id' => 2, 'nom' => 'Maison', 'description' => 'Maisons individuelles'],
                ['id' => 3, 'nom' => 'Villa', 'description' => 'Villas de luxe'],
                ['id' => 4, 'nom' => 'Terrain', 'description' => 'Terrains constructibles'],
                ['id' => 5, 'nom' => 'Local Commercial', 'description' => 'Locaux commerciaux'],
                ['id' => 6, 'nom' => 'Bureau', 'description' => 'Espaces de bureau']
            ];
        }
        
        // Wilayas d'Algérie
        $this->wilayasAlgerie = [
            ['code' => '01', 'nom' => 'Adrar'],
            ['code' => '02', 'nom' => 'Chlef'],
            ['code' => '03', 'nom' => 'Laghouat'],
            ['code' => '04', 'nom' => 'Oum El Bouaghi'],
            ['code' => '05', 'nom' => 'Batna'],
            ['code' => '06', 'nom' => 'Béjaïa'],
            ['code' => '07', 'nom' => 'Biskra'],
            ['code' => '08', 'nom' => 'Béchar'],
            ['code' => '09', 'nom' => 'Blida'],
            ['code' => '10', 'nom' => 'Bouira'],
            ['code' => '11', 'nom' => 'Tamanrasset'],
            ['code' => '12', 'nom' => 'Tébessa'],
            ['code' => '13', 'nom' => 'Tlemcen'],
            ['code' => '14', 'nom' => 'Tiaret'],
            ['code' => '15', 'nom' => 'Tizi Ouzou'],
            ['code' => '16', 'nom' => 'Alger'],
            ['code' => '17', 'nom' => 'Djelfa'],
            ['code' => '18', 'nom' => 'Jijel'],
            ['code' => '19', 'nom' => 'Sétif'],
            ['code' => '20', 'nom' => 'Saïda'],
            ['code' => '21', 'nom' => 'Skikda'],
            ['code' => '22', 'nom' => 'Sidi Bel Abbès'],
            ['code' => '23', 'nom' => 'Annaba'],
            ['code' => '24', 'nom' => 'Guelma'],
            ['code' => '25', 'nom' => 'Constantine'],
            ['code' => '26', 'nom' => 'Médéa'],
            ['code' => '27', 'nom' => 'Mostaganem'],
            ['code' => '28', 'nom' => 'M\'Sila'],
            ['code' => '29', 'nom' => 'Mascara'],
            ['code' => '30', 'nom' => 'Ouargla'],
            ['code' => '31', 'nom' => 'Oran'],
            ['code' => '32', 'nom' => 'El Bayadh'],
            ['code' => '33', 'nom' => 'Illizi'],
            ['code' => '34', 'nom' => 'Bordj Bou Arréridj'],
            ['code' => '35', 'nom' => 'Boumerdès'],
            ['code' => '36', 'nom' => 'El Tarf'],
            ['code' => '37', 'nom' => 'Tindouf'],
            ['code' => '38', 'nom' => 'Tissemsilt'],
            ['code' => '39', 'nom' => 'El Oued'],
            ['code' => '40', 'nom' => 'Khenchela'],
            ['code' => '41', 'nom' => 'Souk Ahras'],
            ['code' => '42', 'nom' => 'Tipaza'],
            ['code' => '43', 'nom' => 'Mila'],
            ['code' => '44', 'nom' => 'Aïn Defla'],
            ['code' => '45', 'nom' => 'Naâma'],
            ['code' => '46', 'nom' => 'Aïn Témouchent'],
            ['code' => '47', 'nom' => 'Ghardaïa'],
            ['code' => '48', 'nom' => 'Relizane']
        ];
    }

    private function getAnnoncesRecentesManual($conn) {
        try {
            $query = "
                SELECT 
                    a.id,
                    a.titre,
                    a.prix,
                    a.surface,
                    a.chambres,
                    a.salles_bain,
                    a.ville,
                    a.type_transaction,
                    a.date_creation,
                    tp.nom as type_propriete
                FROM annonces a
                LEFT JOIN types_proprietes tp ON a.type_propriete_id = tp.id
                WHERE a.actif = 1
                ORDER BY a.date_creation DESC
                LIMIT 6
            ";
            
            $conn->exec($query);
            return $conn->data;
            
        } catch (\Exception $e) {
            error_log("Erreur lors de la récupération des annonces récentes: " . $e->getMessage());
            return [];
        }
    }
    
    public function showHomePage()
    {
        $viewData = [
            'typesProprietes' => $this->typesProprietes,
            'wilayasAlgerie' => $this->wilayasAlgerie,
            'villesAvecAnnonces' => $this->villesAvecAnnonces,
            'annoncesRecentes' => $this->annoncesRecentes,
            'pathPrefix' => $this->pathPrefix,
            'publicIndex' => $this  // Passer l'instance de la classe pour utiliser les méthodes
        ];
        
        // Inclure la vue de la page d'accueil
        include 'views/PublicHomeHtml.php';
    }
    
    public function showLoginPage()
    {
        $viewData = [
            'pathPrefix' => $this->pathPrefix
        ];
        
        // Inclure la vue de la page de connexion
        include 'views/PublicLoginHtml.php';
    }
    
    public function showRegisterPage()
    {
        $viewData = [
            'pathPrefix' => $this->pathPrefix
        ];
        
        // Inclure la vue de la page d'inscription
        include 'views/PublicRegisterHtml.php';
    }
    
    public function processPageRequest()
    {
        if (isset($_GET['page'])) {
            switch ($_GET['page']) {
                case 'login':
                case 'log':
                    $this->showLoginPage();
                    break;
                    
                case 'register':
                    $this->showRegisterPage();
                    break;
                
                case 'search':
                    $this->showSearchPage();
                    break;
                
                case 'annonce':
                    $this->showAnnoncePage();
                    break;
                    
                default:
                    $this->showHomePage();
                    break;
            }
        } else {
            $this->showHomePage();
        }
    }
    
    /**
     * Afficher la page de recherche
     */
    private function showSearchPage()
    {
        // Inclure la connexion à la base de données
        include_once 'Parametre/taghilt/taghilt.php';
        include_once 'Parametre/taghilt/LoadDirClass.php';
        include_once 'Parametre/taghilt/dbConnect.php';
        
        $conn = dbConnect(true); // Activer le mode verbose pour voir les erreurs
        
        if (!$conn) {
            die('❌ Erreur : Impossible de se connecter à la base de données. Vérifiez la configuration dans config.php');
        }
        
        // Inclure la page d'initialisation de la recherche
        include_once 'SitePage/searchInit.php';
    }
    
    /**
     * Afficher la page de détails d'une annonce
     */
    private function showAnnoncePage()
    {
        // Inclure la connexion à la base de données
        include_once 'Parametre/taghilt/taghilt.php';
        include_once 'Parametre/taghilt/LoadDirClass.php';
        include_once 'Parametre/taghilt/dbConnect.php';
        
        $conn = dbConnect(true); // Activer le mode verbose
        
        if (!$conn) {
            die('❌ Erreur : Impossible de se connecter à la base de données. Vérifiez la configuration dans config.php');
        }
        
        // Inclure la page d'initialisation de l'annonce
        include_once 'SitePage/annonceDetailInit.php';
    }
    
    /**
     * Génère une carte de propriété HTML
     */
    public function generatePropertyCard($annonce) {
        // Déterminer l'image en fonction du type de propriété
        $typeImage = $this->getPropertyTypeImage($annonce['type_propriete'] ?? '');
        
        // Déterminer la couleur du badge selon le type de transaction
        $badgeClass = $this->getTransactionBadgeClass($annonce['type_transaction'] ?? 'vente');
        
        // Générer les caractéristiques
        $features = $this->generatePropertyFeatures($annonce);
        
        $html = '
        <div class="col-lg-4 col-md-6 mb-4" style="overflow: hidden; max-width: 100%; box-sizing: border-box;">
            <div class="property-card h-100" style="overflow: hidden; max-width: 100%; box-sizing: border-box;">
                <div class="property-image" style="overflow: hidden; max-width: 100%;">
                    <img src="' . $this->pathPrefix . 'assets/images/' . $typeImage . '" 
                         alt="' . htmlspecialchars($annonce['titre'] ?? 'Propriété') . '" 
                         class="img-fluid"
                         style="max-width: 100%; height: 200px; object-fit: cover;"
                         onerror="this.src=\'' . $this->pathPrefix . 'assets/images/property-placeholder.svg\';">
                    <div class="property-badge">
                        <span class="badge ' . $badgeClass . '">' . htmlspecialchars($annonce['type_transaction'] ?? 'Vente') . '</span>
                    </div>
                </div>
                <div class="property-content" style="padding: 1.5rem; overflow: hidden; max-width: 100%; box-sizing: border-box; display: flex; flex-direction: column; height: 100%;">
                    <div class="property-info" style="flex-grow: 1;">
                        <h5 style="overflow: hidden; text-overflow: ellipsis; white-space: nowrap; max-width: 100%; margin-bottom: 1rem;">' . htmlspecialchars($annonce['titre'] ?? 'Titre non disponible') . '</h5>
                        <p class="property-location" style="overflow: hidden; text-overflow: ellipsis; white-space: nowrap; max-width: 100%; color: #6c757d; font-size: 0.9rem; margin-bottom: 1rem;">
                            <i class="fas fa-map-marker-alt me-2"></i>
                            ' . htmlspecialchars($annonce['ville'] ?? 'Localisation non précisée') . '
                        </p>
                    </div>
                    <div class="property-bottom" style="margin-top: auto;">
                        <div class="property-features" style="display: flex; gap: 1rem; flex-wrap: wrap; max-width: 100%; overflow: hidden; font-size: 0.85rem; color: #6c757d;">' . $features . '</div>
                    </div>
                </div>
            </div>
        </div>';
        
        return $html;
    }
    
    /**
     * Détermine l'image à utiliser selon le type de propriété
     */
    private function getPropertyTypeImage($typeProperty) {
        $images = [
            'appartement' => 'appartement.svg',
            'maison' => 'maison.svg',
            'villa' => 'villa.svg',
            'terrain' => 'terrain.svg',
            'local commercial' => 'local_commercial.svg',
            'garage' => 'garage.svg'
        ];
        
        $typeLower = strtolower($typeProperty);
        return $images[$typeLower] ?? 'property-placeholder.svg';
    }
    
    /**
     * Détermine la classe CSS du badge selon le type de transaction
     */
    private function getTransactionBadgeClass($transaction) {
        $classes = [
            'vente' => 'bg-primary',
            'location' => 'bg-success',
            'terrain' => 'bg-warning'
        ];
        
        $transactionLower = strtolower($transaction);
        return $classes[$transactionLower] ?? 'bg-primary';
    }
    
    /**
     * Génère les caractéristiques de la propriété
     */
    private function generatePropertyFeatures($annonce) {
        $features = [];
        
        if (!empty($annonce['chambres']) && $annonce['chambres'] > 0) {
            $features[] = '<span><i class="fas fa-bed me-1"></i>' . $annonce['chambres'] . '</span>';
        }
        
        if (!empty($annonce['salles_bain']) && $annonce['salles_bain'] > 0) {
            $features[] = '<span><i class="fas fa-bath me-1"></i>' . $annonce['salles_bain'] . '</span>';
        }
        
        if (!empty($annonce['surface']) && $annonce['surface'] > 0) {
            $features[] = '<span><i class="fas fa-expand-arrows-alt me-1"></i>' . $annonce['surface'] . 'm²</span>';
        }
        
        // Si c'est un terrain et pas d'autres caractéristiques
        if (empty($features) && stripos($annonce['type_propriete'] ?? '', 'terrain') !== false) {
            $features[] = '<span><i class="fas fa-map me-1"></i>Zone constructible</span>';
        }
        
        return implode('', $features);
    }
}