<?php
namespace Classe;

/**
 * Classe pour gérer les permissions et contrôles d'accès
 */
class PermissionChecker
{
    /**
     * Vérifier si l'utilisateur a le niveau d'accès requis
     * 
     * @param int $requiredLevel Niveau d'accès requis (10, 20, 40, 60, 80, 100)
     * @param string $NomSite Nom du site pour les sessions
     * @return bool True si l'utilisateur a l'accès, False sinon
     */
    public static function hasAccess($requiredLevel, $NomSite)
    {
        if (!isset($_SESSION[$NomSite]['userRoles']) || empty($_SESSION[$NomSite]['userRoles'])) {
            return false;
        }

        $userMaxLevel = 0;
        $roles = $_SESSION[$NomSite]['userRoles'];
        
        foreach ($roles as $role) {
            if (isset($role['niveau_acces'])) {
                $userMaxLevel = max($userMaxLevel, (int)$role['niveau_acces']);
            }
            
            // Compatibilité avec l'ancien système de noms
            if (isset($role['role_name'])) {
                switch ($role['role_name']) {
                    case 'Super Administrateur':
                        $userMaxLevel = max($userMaxLevel, 100);
                        break;
                    case 'Administrateur':
                        $userMaxLevel = max($userMaxLevel, 80);
                        break;
                    case 'Modérateur':
                        $userMaxLevel = max($userMaxLevel, 60);
                        break;
                    case 'Agent Premium':
                        $userMaxLevel = max($userMaxLevel, 50);
                        break;
                    case 'Agent Standard':
                        $userMaxLevel = max($userMaxLevel, 40);
                        break;
                    case 'Particulier':
                        $userMaxLevel = max($userMaxLevel, 20);
                        break;
                    case 'Visiteur':
                        $userMaxLevel = max($userMaxLevel, 10);
                        break;
                }
            }
        }

        return $userMaxLevel >= $requiredLevel;
    }

    /**
     * Rediriger vers la page d'accès refusé si l'utilisateur n'a pas les permissions
     * 
     * @param int $requiredLevel Niveau d'accès requis
     * @param string $NomSite Nom du site pour les sessions
     * @param string $currentPage Page actuelle (optionnel)
     */
    public static function requireAccess($requiredLevel, $NomSite, $currentPage = '')
    {
        if (!self::hasAccess($requiredLevel, $NomSite)) {
            // Sauvegarder la page actuelle
            if (!empty($currentPage)) {
                $_SESSION[$NomSite]['lastPage'] = $currentPage;
            }
            
            // Afficher directement la page d'accès refusé
            self::showAccessDeniedPage($requiredLevel, $NomSite);
            exit();
        }
    }

    /**
     * Obtenir le niveau d'accès maximum de l'utilisateur
     * 
     * @param string $NomSite Nom du site pour les sessions
     * @return int Niveau d'accès maximum
     */
    public static function getUserLevel($NomSite)
    {
        if (!isset($_SESSION[$NomSite]['userRoles']) || empty($_SESSION[$NomSite]['userRoles'])) {
            return 0;
        }

        $userMaxLevel = 0;
        $roles = $_SESSION[$NomSite]['userRoles'];
        
        foreach ($roles as $role) {
            if (isset($role['niveau_acces'])) {
                $userMaxLevel = max($userMaxLevel, (int)$role['niveau_acces']);
            }
        }

        return $userMaxLevel;
    }

    /**
     * Obtenir les noms des rôles de l'utilisateur
     * 
     * @param string $NomSite Nom du site pour les sessions
     * @return string Noms des rôles séparés par des virgules
     */
    public static function getUserRoles($NomSite)
    {
        if (!isset($_SESSION[$NomSite]['userRoles']) || empty($_SESSION[$NomSite]['userRoles'])) {
            return 'Aucun rôle';
        }

        $roles = $_SESSION[$NomSite]['userRoles'];
        $roleNames = array_map(function($role) {
            return $role['role_name'] ?? 'Inconnu';
        }, $roles);

        return implode(', ', $roleNames);
    }

    /**
     * Affiche directement la page d'accès refusé sans redirection
     */
    public static function showAccessDeniedPage($requiredLevel, $NomSite)
    {
        // Inclure les classes nécessaires
        include_once(__DIR__ . "/../views/VariousViewsHtml.php");
        include_once(__DIR__ . "/../Classe/FunctionClasse.php");
        include(__DIR__ . "/../Parametre/" . getPath(1) . "/" . getPath(1) . ".php");
        
        // Récupérer les informations de l'utilisateur
        global $NomSiteCap;
        $userRole = '';
        
        if (isset($_SESSION[$NomSite]['userRoles']) && !empty($_SESSION[$NomSite]['userRoles'])) {
            $roles = $_SESSION[$NomSite]['userRoles'];
            $roleNames = array_map(function($role) {
                return $role['role_name'] ?? 'Inconnu';
            }, $roles);
            $userRole = implode(', ', $roleNames);
        }
        
        // Afficher la page d'accès refusé
        $variousViews = new \Views\VariousViewsHtml();
        $variousViews->generateHeader(false, false, "Accès refusé - " . $NomSiteCap, "normal");
        $variousViews->showAccessDenied($userRole, (int)$requiredLevel);
        
        // Script de redirection automatique
        $lastPage = $_SESSION[$NomSite]['lastPage'] ?? 'index.php';
        echo "<script>
        setTimeout(function() {
            window.location.href = '$lastPage';
        }, 5000);
        </script>";
        
        $variousViews->generateFooter(false, false, "normal");
    }
}
?>